const TelegramBot = require('node-telegram-bot-api');
const mysql = require('mysql2/promise');
const fs = require('fs');

// Configuration - इसे अपने actual credentials से replace करें
const config = {
  telegram: {
    token: '7505738765:AAFOLhlZUrqDyrbD5JWN_p9KC2MnBIMZqAg'
  },
  database: {
    host: 'localhost',
    user: 'mafiaapi_panel',
    password: 'mafiaapi_panel',
    database: 'mafiaapi_panel',
    connectionLimit: 10,
    acquireTimeout: 60000,
    timeout: 60000
  }
};

// Initialize bot with polling
const bot = new TelegramBot(config.telegram.token, {
  polling: true,
  request: {
    timeout: 60000,
    agentOptions: {
      keepAlive: true,
      maxSockets: Infinity
    }
  }
});

// Database connection pool with better memory management
let dbPool;
(async () => {
  try {
    dbPool = await mysql.createPool({
      ...config.database,
      waitForConnections: true,
      queueLimit: 0
    });
    console.log('✅ Connected to database successfully');
  } catch (error) {
    console.error('❌ Database connection failed:', error.message);
    process.exit(1);
  }
})();

// User states for custom key generation
const userStates = new Map();

// Generate key in format USERNAME-DAYS-RANDOM
function generateKey(username, duration, randomLength = 5) {
  const characters = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
  let randomPart = '';
  for (let i = 0; i < randomLength; i++) {
    randomPart += characters[Math.floor(Math.random() * characters.length)];
  }
  return `${username.toUpperCase()}-${duration}-${randomPart}`;
}

// Generate custom key with provided prefix
function generateCustomKey(prefix, duration) {
  const characters = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
  let randomPart = '';
  for (let i = 0; i < 5; i++) {
    randomPart += characters[Math.floor(Math.random() * characters.length)];
  }
  return `${prefix.toUpperCase()}-${duration}-${randomPart}`;
}

// Memory management function
function freeMemory() {
  if (global.gc) {
    global.gc();
  }
}

// Log errors to file
function logError(error, context = '') {
  const timestamp = new Date().toISOString();
  const logMessage = `${timestamp} - ${context}: ${error.message || error}\n`;
  fs.appendFile('error.log', logMessage, (err) => {
    if (err) console.error('❌ Failed to write to error log:', err);
  });
  console.error(`❌ Error in ${context}:`, error);
}

// Clean up resources periodically
setInterval(() => {
  freeMemory();
  // Clear old user states (older than 1 hour)
  const now = Date.now();
  for (const [chatId, state] of userStates.entries()) {
    if (now - state.timestamp > 3600000) { // 1 hour
      userStates.delete(chatId);
    }
  }
}, 300000); // Every 5 minutes

// Main bot logic
bot.on('message', async (msg) => {
  const chatId = msg.chat.id;
  const text = msg.text || '';
  const firstName = msg.chat.first_name || 'User';
  const telegramUsername = msg.chat.username || '';

  try {
    // Check if user exists in database
    const [users] = await dbPool.execute(
      'SELECT * FROM users WHERE email = ? OR username = ? LIMIT 1',
      [chatId.toString(), telegramUsername]
    );

    if (!users || users.length === 0) {
      await bot.sendMessage(chatId, '⛔ You don\'t have permission to use this bot. Please register first on our website.');
      return;
    }

    const user = users[0];
    const dbUsername = user.username;

    // Update chat ID if not set
    if (!user.email) {
      await dbPool.execute(
        'UPDATE users SET email = ? WHERE username = ?',
        [chatId.toString(), dbUsername]
      );
    }

    // Handle custom key prefix input
    if ((userStates.has(chatId) && userStates.get(chatId).action === 'awaiting_custom_key_prefix') || 
        text.startsWith('/custom')) {
      
      let prefix = text.replace('/custom', '').trim();
      
      if (!prefix) {
        await bot.sendMessage(chatId, '⚠️ Please provide a prefix for your custom key.\n\nExample: <code>/custom VIP</code> or <code>/custom YOURNAME</code>', { parse_mode: 'HTML' });
        return;
      }
      
      // Clean up the prefix
      prefix = prefix.replace(/[^A-Za-z0-9]/g, '');
      prefix = prefix.substring(0, 10);
      
      if (!prefix) {
        await bot.sendMessage(chatId, '⚠️ Please provide a valid prefix (letters and numbers only).');
        return;
      }
      
      // Create duration selection keyboard
      const keyboard = {
        inline_keyboard: [
          [
            { text: '1 Day', callback_data: `custom_duration_1_${prefix}` },
            { text: '3 Days', callback_data: `custom_duration_3_${prefix}` }
          ],
          [
            { text: '7 Days', callback_data: `custom_duration_7_${prefix}` },
            { text: '14 Days', callback_data: `custom_duration_14_${prefix}` }
          ],
          [
            { text: '30 Days', callback_data: `custom_duration_30_${prefix}` },
            { text: '60 Days', callback_data: `custom_duration_60_${prefix}` }
          ]
        ]
      };
      
      await bot.sendMessage(chatId, `📅 Please select duration for your custom key <b>${prefix}</b>:`, {
        parse_mode: 'HTML',
        reply_markup: keyboard
      });
      
      // Clear the user state if it was set
      if (userStates.has(chatId)) {
        userStates.delete(chatId);
      }
      
      return;
    }
    
    // Handle commands
    if (text.startsWith('/start')) {
      const welcomeImage = 'https://i.postimg.cc/907B43T5/file-0000000004f061faac577cf523b54d95.png';
      let caption = `👋 Welcome to ${firstName}!\n\n`;
      caption += `🆔 Your Telegram ID: <code>${chatId}</code>\n`;
      caption += `👤 Username: @${telegramUsername}\n`;
      caption += `📋 Panel Account: ${dbUsername}\n\n`;
      caption += '⚡ <b>Developer:</b> @MAFIA_HACKERi\n\n';
      caption += 'Click below to generate a new key:';
      
      // Create inline keyboard for key generation
      const keyboard = {
        inline_keyboard: [
          [
            { text: '🔑 Generate Key', callback_data: 'generate_key' },
            { text: '🔠 Custom Key', callback_data: 'custom_key' }
          ],
          [
            { text: '🗑️ Delete Key', callback_data: 'delete_key' },
            { text: '🌐 Panel Link', callback_data: 'panel_link' }
          ]
        ]
      };
      
      // Send the welcome image with caption and buttons
      await bot.sendPhoto(chatId, welcomeImage, {
        caption: caption,
        parse_mode: 'HTML',
        reply_markup: keyboard
      });
    }
    
    // Handle delete key command
    if (text.startsWith('/delete')) {
      const keyToDelete = text.replace('/delete', '').trim();
      
      if (!keyToDelete) {
        await bot.sendMessage(chatId, '⚠️ Please provide a key to delete. Usage: /delete KEY-HERE');
        return;
      }
      
      try {
        // Check if key exists and belongs to the user
        const [keys] = await dbPool.execute(
          'SELECT * FROM keys_code WHERE user_key = ? AND registrator = ? LIMIT 1',
          [keyToDelete, dbUsername]
        );
        
        if (!keys || keys.length === 0) {
          await bot.sendMessage(chatId, '⛔ Key not found or you don\'t have permission to delete it.');
          return;
        }
        
        // Delete the key
        await dbPool.execute(
          'DELETE FROM keys_code WHERE user_key = ?',
          [keyToDelete]
        );
        
        await bot.sendMessage(chatId, `✅ Key <code>${keyToDelete}</code> has been successfully deleted.`, {
          parse_mode: 'HTML'
        });
        
      } catch (error) {
        logError(error, 'Delete key');
        await bot.sendMessage(chatId, '⚠️ Error deleting key. Please try again.');
      }
    }
    
    // Handle help command
    if (text.startsWith('/help')) {
      let helpMessage = '🤖 <b>Bot Commands</b>\n\n';
      helpMessage += '🔑 <code>/start</code> - Show main menu\n';
      helpMessage += '🔠 <code>/custom PREFIX</code> - Create custom key (e.g., <code>/custom VIP</code>)\n';
      helpMessage += '🗑️ <code>/delete KEY</code> - Delete a key\n';
      helpMessage += 'ℹ️ <code>/help</code> - Show this help message\n\n';
      helpMessage += 'You can also use the buttons in the menu for these actions.';
      
      await bot.sendMessage(chatId, helpMessage, { parse_mode: 'HTML' });
    }
    
  } catch (error) {
    logError(error, 'Message handler');
    await bot.sendMessage(chatId, '⚠️ An error occurred. Please try again later.');
  }
});

// Handle callback queries (button presses)
bot.on('callback_query', async (callbackQuery) => {
  const chatId = callbackQuery.message.chat.id;
  const data = callbackQuery.data;
  const telegramUsername = callbackQuery.from.username || '';

  try {
    // Get the user's database username
    const [users] = await dbPool.execute(
      'SELECT username FROM users WHERE email = ? OR username = ? LIMIT 1',
      [chatId.toString(), telegramUsername]
    );
    
    if (!users || users.length === 0) {
      await bot.sendMessage(chatId, '⛔ User not found in database.');
      return;
    }
    
    const dbUsername = users[0].username;
    
    if (data === 'generate_key') {
      // Create duration selection keyboard
      const keyboard = {
        inline_keyboard: [
          [
            { text: '1 Day', callback_data: 'duration_1' },
            { text: '3 Days', callback_data: 'duration_3' }
          ],
          [
            { text: '7 Days', callback_data: 'duration_7' },
            { text: '14 Days', callback_data: 'duration_14' }
          ],
          [
            { text: '30 Days', callback_data: 'duration_30' },
            { text: '60 Days', callback_data: 'duration_60' }
          ]
        ]
      };
      
      await bot.sendMessage(chatId, '📅 Please select key duration:', {
        reply_markup: keyboard
      });
    }
    
    if (data === 'custom_key') {
      // Set user state to awaiting custom key prefix
      userStates.set(chatId, {
        action: 'awaiting_custom_key_prefix',
        timestamp: Date.now()
      });
      
      await bot.sendMessage(chatId, '✏️ Please enter your desired key prefix (e.g., VIP, PREMIUM, YOURNAME) or use the command:\n\n<code>/custom PREFIX</code>\n\nExample: <code>/custom VIP</code>', {
        parse_mode: 'HTML'
      });
    }
    
    if (data === 'delete_key') {
      await bot.sendMessage(chatId, '🗑️ To delete a key, send the command:\n\n<code>/delete KEY-HERE</code>\n\nExample: <code>/delete USER-7-ABC12</code>', {
        parse_mode: 'HTML'
      });
    }
    
    if (data === 'panel_link') {
      const panelUrl = 'https://powerof.mafia-api.site';
      let message = '🌐 <b>VIP Panel Link</b>\n\n';
      message += `<a href="${panelUrl}">${panelUrl}</a>\n\n`;
      message += 'Login Your Vip panel';
      
      await bot.sendMessage(chatId, message, {
        parse_mode: 'HTML'
      });
    }
    
    // Handle duration selection for regular keys
    if (data.startsWith('duration_') && !data.startsWith('custom_duration_')) {
      const duration = parseInt(data.replace('duration_', ''));
      const key = generateKey(dbUsername, duration);
      const expirationDate = new Date();
      expirationDate.setDate(expirationDate.getDate() + duration);
      const currentDate = new Date();
      
      try {
        // Insert key into keys_code table
        await dbPool.execute(
          `INSERT INTO keys_code 
          (game, user_key, duration, expired_date, max_devices, status, registrator, created_at, updated_at) 
          VALUES (?, ?, ?, ?, ?, 1, ?, ?, ?)`,
          [
            'BGMI',
            key,
            duration,
            expirationDate,
            1,
            dbUsername,
            currentDate,
            currentDate
          ]
        );
        
        let response = '✅ <b>Key Generated Successfully!</b>\n\n';
        response += '🎮 Game: <b>BGMI</b>\n';
        response += `🔑 Key: <code>${key}</code>\n`;
        response += `⏳ Duration: <b>${duration} Days</b>\n`;
        response += `📅 Expires: <b>${expirationDate.toLocaleDateString()}</b>\n`;
        response += `👤 Registrator: <b>${dbUsername}</b>\n`;
        response += '📱 Max Devices: <b>1</b>\n\n';
        response += 'Thank you for using our service!';
        
        await bot.sendMessage(chatId, response, {
          parse_mode: 'HTML'
        });
        
      } catch (error) {
        logError(error, 'Generate key');
        await bot.sendMessage(chatId, '⚠️ Error generating key. Please try again.');
      }
    }
    
    // Handle duration selection for custom keys
    if (data.startsWith('custom_duration_')) {
      const parts = data.split('_');
      const duration = parseInt(parts[2]);
      const prefix = parts[3];
      
      const key = generateCustomKey(prefix, duration);
      const expirationDate = new Date();
      expirationDate.setDate(expirationDate.getDate() + duration);
      const currentDate = new Date();
      
      try {
        // Insert key into keys_code table
        await dbPool.execute(
          `INSERT INTO keys_code 
          (game, user_key, duration, expired_date, max_devices, status, registrator, created_at, updated_at) 
          VALUES (?, ?, ?, ?, ?, 1, ?, ?, ?)`,
          [
            'BGMI',
            key,
            duration,
            expirationDate,
            1,
            dbUsername,
            currentDate,
            currentDate
          ]
        );
        
        let response = '✅ <b>Custom Key Generated Successfully!</b>\n\n';
        response += '🎮 Game: <b>BGMI</b>\n';
        response += `🔑 Key: <code>${key}</code>\n`;
        response += `⏳ Duration: <b>${duration} Days</b>\n`;
        response += `📅 Expires: <b>${expirationDate.toLocaleDateString()}</b>\n`;
        response += `👤 Registrator: <b>${dbUsername}</b>\n`;
        response += '📱 Max Devices: <b>1</b>\n\n';
        response += 'Thank you for using our service!';
        
        await bot.sendMessage(chatId, response, {
          parse_mode: 'HTML'
        });
        
      } catch (error) {
        logError(error, 'Custom key');
        await bot.sendMessage(chatId, '⚠️ Already exists key trying to other custom key');
      }
    }
    
    // Answer the callback query to remove the loading state
    await bot.answerCallbackQuery(callbackQuery.id);
    
  } catch (error) {
    logError(error, 'Callback query');
    await bot.sendMessage(chatId, '⚠️ An error occurred. Please try again later.');
  }
});

// Error handling
bot.on('error', (error) => {
  logError(error, 'Telegram Bot');
});

bot.on('polling_error', (error) => {
  logError(error, 'Telegram Polling');
});

process.on('unhandledRejection', (error) => {
  logError(error, 'Unhandled Promise Rejection');
});

process.on('uncaughtException', (error) => {
  logError(error, 'Uncaught Exception');
  process.exit(1);
});

// Graceful shutdown
process.on('SIGINT', async () => {
  console.log('🛑 Shutting down bot gracefully...');
  if (dbPool) {
    await dbPool.end();
  }
  bot.stopPolling();
  process.exit(0);
});

console.log('✅ Bot is running with memory optimization...');